// Copyright 2025 Google, Inc. All rights reserved.
//
// Use of this source code is governed by a BSD-style license
// that can be found in the LICENSE file in the root of the source
// tree.
// Copyright 2025 The GoPacket Authors. All rights reserved.
//
// Use of this source code is governed by a BSD-style license that can be found
// in the LICENSE file in the root of the source tree.
package layers

import (
	"encoding/hex"
	"net"
	"testing"

	"github.com/gopacket/gopacket"
)

func TestGueDecoding(t *testing.T) {
	// This is a packet generated by the Linux kernel GUE implementation, captured
	// by pcap. It includes:
	// - Ethernet
	// - IPv4
	// - UDP
	// - AGueVar0 (port 666)
	// - IPv4
	// - ICMP (ping)
	// TODO: build a test packet using port 666 (0x029a)
	ph := `02427b2522f502420ae0d90608004500007451ea4000ff119d050ae0d9060afa9da88c0f029a00608cfa000400004500005459f240004001e2cd0ae0d9060afd0f0608000a7e005f000cea811f59000000005daa080000000000000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f2021222324252627`
	pr, err := hex.DecodeString(ph)
	if err != nil {
		t.Errorf("Error decoding hex packet: %v", err)
	}
	p := gopacket.NewPacket(pr, LayerTypeEthernet, gopacket.Default)
	// require.Nil(t, p.ErrorLayer())
	if p.ErrorLayer() != nil {
		t.Errorf("AGUEVar0 layer is nil")
	}
	t.Logf("%v", p)
	gue := p.Layer(LayerTypeAGUEVar0).(AGUEVar0)
	// require.NotNil(t, gue)
	if uint8(0) != gue.Version {
		t.Errorf("gue.Version is not 0")
	}
}

var testIPv4OverAGUEVar0 = []gopacket.SerializableLayer{
	&Ethernet{
		SrcMAC:       net.HardwareAddr{142, 122, 18, 195, 169, 113},
		DstMAC:       net.HardwareAddr{58, 86, 107, 105, 89, 94},
		EthernetType: EthernetTypeIPv4,
	},
	&IPv4{
		Version:  4,
		SrcIP:    net.IP{192, 168, 1, 1},
		DstIP:    net.IP{192, 168, 1, 2},
		Protocol: IPProtocolUDP,
		Flags:    IPv4DontFragment,
		TTL:      64,
		Id:       33852,
		IHL:      5,
	},
	&UDP{
		SrcPort: 8,
		DstPort: 666,
	},
	&AGUEVar0{
		Protocol: IPProtocolIPv4,
	},
	&IPv4{
		Version:  4,
		SrcIP:    net.IP{172, 16, 1, 1},
		DstIP:    net.IP{172, 16, 2, 1},
		Protocol: IPProtocolICMPv4,
		Flags:    IPv4DontFragment,
		TTL:      64,
		IHL:      5,
		Id:       1160,
	},
	&ICMPv4{
		TypeCode: CreateICMPv4TypeCode(ICMPv4TypeEchoRequest, 0),
		Id:       4724,
		Seq:      1,
	},
	gopacket.Payload{
		0xc8, 0x92, 0xa3, 0x54, 0x00, 0x00, 0x00, 0x00, 0x38, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
		0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
		0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
	},
}

func TestIPv4OverAGUEVar0Encode(t *testing.T) {
	b := gopacket.NewSerializeBuffer()
	opts := gopacket.SerializeOptions{
		ComputeChecksums: false, // if desired, see gre_test:setNetworkLayer()
		FixLengths:       true,
	}
	if err := gopacket.SerializeLayers(b, opts, testIPv4OverAGUEVar0...); err != nil {
		t.Errorf("Unable to serialize: %v", err)
	}
	p := gopacket.NewPacket(b.Bytes(), LinkTypeEthernet, gopacket.Default)
	if p.ErrorLayer() != nil {
		t.Error("Failed to decode packet:", p.ErrorLayer().Error())
	}
	checkLayers(p, []gopacket.LayerType{LayerTypeEthernet, LayerTypeIPv4, LayerTypeUDP, LayerTypeAGUEVar0, LayerTypeIPv4, LayerTypeICMPv4, gopacket.LayerTypePayload}, t)
	// We don't have a corresponding sample packet capture, but if we did, the verify would look like this:
	// if got, want := b.Bytes(), testPacketAGUEVar0``; !reflect.DeepEqual(want, got) {
	// 	t.Errorf("Encoding mismatch, \nwant: %v\ngot %v\n", want, got)
	// }
}
